package main

import (
	"fmt"
	"sync"
)

type Lock struct {
	ch chan bool
}

// Passing the lock by value is not an issue because 
// the structure contains a channel, which is a reference type
func NewLock() Lock {
	lock := Lock{make(chan bool, 1)}
	lock.ch <- true // send
	return lock
}

func (l *Lock) Lock() {
	<-l.ch // receive
}

func (l *Lock) Unlock() {
	l.ch <- true // send
}

var Counters = map[string]int{"a": 0, "b": 0}

func doIncrement(name string, n int, l Lock, wg *sync.WaitGroup) {
	defer wg.Done()
	for i := 0; i < n; i++ {
		l.Lock()
		Counters[name]++
		l.Unlock()
	}
}

func main() {

	my_lock := NewLock()

	var wg sync.WaitGroup

	wg.Add(20)	// Stress test
	// Run goroutines concurrently, all accessing the shared Counters
    for i := 0; i < 10; i++ {
        go doIncrement("a", 1000, my_lock, &wg)
        go doIncrement("b", 500, my_lock, &wg)
    }

	// Wait for the goroutines to finish
	wg.Wait()
	fmt.Println(Counters)

    // Check if it does not fail
    expectedA := 10000
    expectedB := 5000
    if Counters["a"] != expectedA || Counters["b"] != expectedB {
        fmt.Printf("Counters are incorrect: a=%d, b=%d", Counters["a"], Counters["b"])
    }

}