package main

import (
	"fmt"
	"log"
	"net/rpc"
	"os"
	"path/filepath"
	"strconv"

	"gorpc_arith/arith" // Service package path
)

func main() {

	addr := "localhost:" + "12345" // RPC server address and port

	// Connect to RPC server
	client, err := rpc.Dial("tcp", addr)
	if err != nil {
		log.Fatal("Dial error: ", err)
	}
	defer client.Close()

	// Ensure enough command-line arguments are provided
	if len(os.Args) < 3 {
		fmt.Printf("Usage: %s <num1> <num2>\n", filepath.Base(os.Args[0]))
		os.Exit(1)
	}
	// Convert command-line arguments to integers
	n1, err := strconv.Atoi(os.Args[1])
	if err != nil {
    	log.Fatalf("Invalid first number: %v", err)
	}
	n2, err := strconv.Atoi(os.Args[2])
	if err != nil {
    	log.Fatalf("Invalid second number: %v", err)
	}

	args := arith.Args{n1, n2}

	// Synchronous RPC call
	var mulReply arith.Result 	// mulReply stores the RPC result
	log.Printf("Synchronous call: Arithmetic.Multiply")
	err = client.Call("Arithmetic.Multiply", args, &mulReply)
	if err != nil {
		log.Fatal("Multiply error: ", err)
	}
	fmt.Printf("Arithmetic.Multiply: %d*%d=%d\n", args.A, args.B, mulReply)

	// Asynchronous RPC call
	log.Printf("Asynchronous call: Arithmetic.Divide")
	divReply := new(arith.Quotient)
	divCall := client.Go("Arithmetic.Divide", args, divReply, nil)

	// Wait for async call to complete
	<-divCall.Done
	if divCall.Error != nil {
		log.Fatal("Divide error: ", divCall.Error)
	}
	fmt.Printf("Arithmetic.Divide: %d/%d=%d (rem=%d)\n", args.A, args.B, divReply.Quo, divReply.Rem)
}
