package main

import (
	"fmt"
	"log"
	"net/rpc"
	"os"
	"path/filepath"
	"strconv"
	"time"

	"gorpc_arith/arith" // Service package path
)

func main() {

	addr := "localhost:" + "12345" // RPC server address and port

	// Connect to RPC server over HTTP
	client, err := rpc.DialHTTP("tcp", addr)
	if err != nil {
		log.Fatal("Dial error: ", err)
	}
	defer client.Close()

	// Ensure enough command-line arguments are provided
	if len(os.Args) < 3 {
		fmt.Printf("Usage: %s <num1> <num2>\n", filepath.Base(os.Args[0]))
		os.Exit(1)
	}
	// Convert command-line arguments to integers
	n1, err := strconv.Atoi(os.Args[1])
	if err != nil {
    	log.Fatalf("Invalid first number: %v", err)
	}
	n2, err := strconv.Atoi(os.Args[2])
	if err != nil {
    	log.Fatalf("Invalid second number: %v", err)
	}

	args := arith.Args{n1, n2}
	// Synchronous RPC call
	var mulReply arith.Result 	// reply stores the RPC result
	log.Printf("Synchronous call: Arithmetic.Multiply")
	err = client.Call("Arithmetic.Multiply", args, &mulReply)
	if err != nil {
		log.Fatal("Multiply error: ", err)
	}
	fmt.Printf("Arithmetic.Multiply: %d*%d=%d\n", args.A, args.B, mulReply)

	// Asynchronous RPC call
	log.Printf("Asynchronous call: Arithmetic.Divide")
	divReply := new(arith.Quotient)
	divCall := client.Go("Arithmetic.Divide", args, divReply, nil)

	// Channel to notify when async call is complete
	done := make(chan bool)

	// Goroutine to handle async result
	go func() {
		<-divCall.Done	// Wait for the async result
		if divCall.Error != nil {
			log.Fatalf("Divide error: %v\n", divCall.Error)
		}
		fmt.Printf("Arithmetic.Divide: %d / %d = %d (remainder = %d)\n", args.A, args.B, divReply.Quo, divReply.Rem)
		done <- true	// Signal the main goroutine that the call completed
		close(done)		// Close the channel after sending
	}()

	// main goroutine can perform other tasks concurrently
	for i := 0; i < 3; i++ {
		fmt.Printf("Performing other tasks... %d\n", i+1)
		time.Sleep(time.Second) // Simulate doing something else
	}

	// Wait for async result to complete
	<-done
	log.Println("Completed other tasks and RPC calls.")
}
