// client.go

package main

import (
  "context"
  "fmt"
  "log"
  "sync"
  "flag"
  "time"

  "google.golang.org/grpc"
  pb "helloworld/helloworld"
)

const (
  defaultName = "world"
)

var (
  addr = flag.String("addr", "localhost:50051", "the address to connect to")
  name = flag.String("name", defaultName, "Name to greet")
)

// Function to call SayHello on the server
func sayHello(client pb.GreeterClient, name string, wg *sync.WaitGroup, errChan chan error) {
  defer wg.Done()

  // Make the gRPC call with a timeout context (1 second)
  ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
  defer cancel() // Ensure that the context is canceled when we are done

  // Call SayHello method with the context
  resp, err := client.SayHello(ctx, &pb.HelloRequest{Name: name})
  if err != nil {
    // If there is an error, send it to the error channel
    errChan <- fmt.Errorf("error calling SayHello for %s: %v", name, err)
    return
  }

  // If no error, print the response
  fmt.Printf("Response from SayHello: %s\n", resp.GetMessage())
}

func main() {
  // Set up the connection to the server
  flag.Parse()  
  conn, err := grpc.Dial(*addr, grpc.WithInsecure(), grpc.WithBlock())
  if err != nil {
    log.Fatalf("did not connect: %v", err)
  }
  defer conn.Close()

  // Create the client for the Greeter service
  client := pb.NewGreeterClient(conn)

  // Use WaitGroup to wait for all concurrent clients to finish
  var wg sync.WaitGroup

  // Create concurrent requests
  names := []string{"Alice", "Bob", "Charlie", "David", "Eve"}

  // Channel to collect errors
  errChan := make(chan error, len(names))

  // Add to the WaitGroup for each name
  wg.Add(len(names))

  // Launch goroutines for each name
  for _, name := range names {
    go sayHello(client, name, &wg, errChan)
  }

  // Wait for all goroutines to finish
  wg.Wait()

  // Close the error channel to stop receiving errors
  close(errChan)

  // Check if there were any errors during the calls
  for err := range errChan {
    log.Printf("Received error: %v", err)
  }

  // Done
  fmt.Println("All SayHello calls have been completed.")
}