package main

import (
    "context"
    "fmt"
    "log"
    "net"
    "sync"
    "time"
    "flag"

    "google.golang.org/grpc"
    pb "helloworld/helloworld"
)

var (
    port = flag.Int("port", 50051, "The server port")
)

// GreeterServer implements the Greeter service
type GreeterServer struct {
    pb.UnimplementedGreeterServer
    state *ServerState
    mu    sync.Mutex
}

// ServerState holds state that persists across gRPC calls
type ServerState struct {
    counter int
}

// NewGreeterServer creates a new instance of GreeterServer
func NewGreeterServer() *GreeterServer {
    return &GreeterServer{
        state: &ServerState{},
    }
}

// SayHello handles the SayHello gRPC request and modifies state
func (s *GreeterServer) SayHello(ctx context.Context, req *pb.HelloRequest) (*pb.HelloReply, error) {
    s.mu.Lock() // Lock for concurrency safety
    defer s.mu.Unlock()

    // Increment the counter on each call
    s.state.counter++

    // Simulate a delay (e.g., doing some work)
    time.Sleep(time.Millisecond * 500)

    // Return a response with the updated state
    return &pb.HelloReply{
        Message: fmt.Sprintf("Hello %s, Counter: %d", req.GetName(), s.state.counter),
    }, nil
}

func main() {
    // Listen for incoming connections
    flag.Parse()
    lis, err := net.Listen("tcp", fmt.Sprintf(":%d", *port))
    if err != nil {
        log.Fatalf("failed to listen: %v", err)
    }

    // Create the server
    server := grpc.NewServer()

    // Register the GreeterServer with the gRPC server
    pb.RegisterGreeterServer(server, NewGreeterServer())
    log.Printf("server listening at %v", lis.Addr())
    if err := server.Serve(lis); err != nil {
        log.Fatalf("failed to serve: %v", err)
    }

    // Start the gRPC server
    fmt.Println("Server listening on port 50051")
    if err := server.Serve(lis); err != nil {
        log.Fatalf("Failed to serve: %v", err)
    }
}
