package main

import (
	"context"
	"log"
	"time"

	"google.golang.org/grpc"
	pb "productinfo/ecommerce" // import package generated from protobuf compiler
)

const (
	address = "localhost:50051"
)

func main() {

	// Set up a connection to the server.
	// Unsecured connection between client and server
	conn, err := grpc.Dial(address, grpc.WithInsecure())
	if err != nil {
		log.Fatalf("did not connect: %v", err)
	}
	defer conn.Close() // Close the connection when everything is done.

	// Pass the connection and create a client stub instance.
	// It contains all the remote methods to invoke the server.
	c := pb.NewProductInfoClient(conn)

	// Prepare to call the server and print out its response.
    name := "Apple iPhone 13"
    description := "The Apple iPhone 13 features a Super Retina XDR display, Ceramic Shield, and improved cameras for stunning photos."
    price := float32(799.00)

    // Optionally, you can use another phone, for example:
    // name := "Samsung Galaxy S21"
    // description := "Samsung Galaxy S21 offers 5G connectivity, a dynamic AMOLED display, and powerful camera capabilities."
    // price := float32(799.99)

	// Create a Context to pass with the remote call.
	// Context object contains metadata (identity of end user, authorization
	// tokens and request deadline) and will exist during the lifetime
	// of the request.
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	// Call addProduct method with product details.
	// addProduct returns a product ID if the action completed successfully.
	// Otherwise it returns an error.
	r, err := c.AddProduct(ctx, &pb.Product{Name: name, Description: description, Price: price})
	if err != nil {
		log.Fatalf("Could not add product: %v", err)
	}
	log.Printf("Product ID: %s added successfully", r.Value)

	// Call getProduct with the product ID.
	// getProduct returns product details if the action completed successfully.
	// Otherwise it returns an error.
	product, err := c.GetProduct(ctx, &pb.ProductID{Value: r.Value})
	if err != nil {
		log.Fatalf("Could not get product: %v", err)
	}
	log.Printf("Product: %v", product.String())
}
