package main

import (
	"context"
	"flag"
	"fmt"
	"log"
	"net"

	"github.com/gofrs/uuid"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	pb "productinfo/ecommerce" // import package generated from protobuf compiler
)

var (
	port = flag.Int("port", 50051, "Server port")
)

// server is used to implement ecommerce/product_info
type server struct {
	pb.UnimplementedProductInfoServer
	productMap map[string]*pb.Product
}

// AddProduct implements ecommerce.AddProduct
// Product and ProductID structs are defined in product_info.pb.go
// Context object contains metadata (identity of end user, authorization
// tokens and request deadline).
// Method has multiple return types.
func (s *server) AddProduct(ctx context.Context,
	in *pb.Product) (*pb.ProductID, error) {
	out, err := uuid.NewV4() // NewV4 returns random generated UUID
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Error while generating Product ID", err)
	}
	in.Id = out.String()
	if s.productMap == nil {
		s.productMap = make(map[string]*pb.Product)
	}
	s.productMap[in.Id] = in
	log.Printf("Product %v : %v - Added.", in.Id, in.Name)
	return &pb.ProductID{Value: in.Id}, status.New(codes.OK, "").Err()
}

// GetProduct implements ecommerce.GetProduct
// Method has multiple return types
func (s *server) GetProduct(ctx context.Context, in *pb.ProductID) (*pb.Product, error) {
	product, exists := s.productMap[in.Value]
	if exists && product != nil {
		log.Printf("Product %v : %v - Retrieved.", product.Id, product.Name)
		return product, status.New(codes.OK, "").Err()
	}
	return nil, status.Errorf(codes.NotFound, "Product does not exist.", in.Value)
}

func main() {

	flag.Parse()

	// Create TCP listener on the port; gRPC server will bind to it
	lis, err := net.Listen("tcp", fmt.Sprintf(":%d", *port))
	if err != nil {
		log.Fatalf("failed to listen: %v", err)
	}

	// Create new gRPC server instance by calling gRPC Go APIs
	s := grpc.NewServer()

	// Register ProductInfo service to the newly created gRPC server
	// by calling generated APIs in product_info_grpc.pb.go
	pb.RegisterProductInfoServer(s, &server{})
	log.Printf("server listening at %v", lis.Addr())

	// Start listening to the incoming messages on the port
	if err := s.Serve(lis); err != nil {
		log.Fatalf("failed to serve: %v", err)
	}
}
