// Consumer that receives messages from a queue managed by RabbitMQ and print them out.
// See https://www.rabbitmq.com/tutorials/tutorial-one-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://godoc.org/github.com/streadway/amqp

package main

import (
	"log"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Fatalf("%s: %s", msg, err)
	}
}

func main() {
	// Connect to RabbitMQ server using amqp.Dial()
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	// To cleanly shutdown by flushing kernel buffers, make sure to close
	// and wait for the response.
	defer conn.Close()

	// Create a channel: all commands are sent over channels. 
	// If any error is returned on a channel, the channel will no longer be valid:
	// throw it away and try with a different channel.
	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	// Declare the queue to hold messages and deliver to consumers
	// Declaring a queue is idempotent: it will only be created if
	// it doesn't exist already
	q, err := ch.QueueDeclare(
		"hello", // name
		false,   // durable: if false, the queue will not survive server restarts
		false,   // autoDelete: delete when unused
		false,   // exclusive
		false,   // noWait
		nil,     // arguments
	)
	failOnError(err, "Failed to declare a queue")

	// Consume queued messages
	msgs, err := ch.Consume(
		q.Name, // queue
		"",     // consumer
		true,   // autoAck: if true, the server will acknowledge delivery to
		// this consumer prior to writing the delivery to the network; in this case,
		// some delivery may get lost if the consumer is unable to process it
		// after the server delivers it
		false, // exclusive: if false, the server will fairly distribute delivery
		// across multiple consumers;
		// if true, this is the sole consumer from this queue
		false, // noLocal (flag unsupported by RabbitMQ)
		false, // noWait: if true, do not wait for RabbitMQ to confirm the request and immediately begin delivery
		nil,   // optional arguments
	)
	failOnError(err, "Failed to register a consumer")

	// Use channel forever to let main() wait, so it does not terminate
	forever := make(chan bool)

	// The RabbitMQ server will push messages asynchronously:
	// read the messages from the channel named msgs (returned by ch.Consume) in a goroutine.
	go func() {
		for d := range msgs {
			log.Printf("Received a message: %s", d.Body)
		}
	}()

	log.Printf(" [*] Waiting for messages. To exit press CTRL+C")
	<-forever
}
