// Producer that sends a single message to a queue managed by RabbitMQ
// See https://www.rabbitmq.com/tutorials/tutorial-one-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://godoc.org/github.com/streadway/amqp

package main

import (
	"context"
	"log"
	"time"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Fatalf("%s: %s", msg, err)
	}
}

func main() {
	// Connect to RabbitMQ server using amqp.Dial()
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	defer conn.Close()

	// Create a channel: all commands are sent over channels.
	// If any error is returned on a channel, the channel will no longer be valid:
	// throw it away and try with a different channel.
	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	// Declare the queue to hold messages and deliver to consumers
	// Declaring a queue is idempotent: it will only be created if
	// it doesn't exist already
	q, err := ch.QueueDeclare(
		"hello", // name
		false,   // durable: if false, the queue will not survive server restarts
		false,   // autoDelete: delete when unused
		false,   // exclusive
		false,   // noWait
		nil,     // arguments
	)
	failOnError(err, "Failed to declare a queue")

	// Define a new context with a given timeout, that will be use by PublishWithContext
	// For context package, see https://pkg.go.dev/context and https://go.dev/blog/context
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	// Cancel context
	defer cancel()

	// Publish a message on the queue
	// Let's publish to the default exchange with the routingKey of the queue name
	// (every queue gets an implicit route to the default exchange)
	body := "Hello World!"
	err = ch.PublishWithContext(ctx,
		"",     // exchange
		q.Name, // routing key
		false,  // mandatory
		false,  // immediate
		amqp.Publishing{
			ContentType: "text/plain",
			Body:        []byte(body),
		})
	log.Printf(" [x] Sent %s", body)
	failOnError(err, "Failed to publish a message")
}
