// Producer that sends tasks to a queue managed by RabbitMQ.
// Goal: distribute time-consuming tasks among multiple workers.
// See https://www.rabbitmq.com/tutorials/tutorial-two-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://pkg.go.dev/github.com/rabbitmq/amqp091-go
// Incremental development - 1st version (v1): send arbitrary messages.
// The number of dots in the message represents the task complexity;
// every dot will account for one second of "work".
// Use this producer with worker_v1.go and worker_v2.go

package main

import (
	"context"
	"log"
	"os"
	"strings"
	"time"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Panicf("%s: %s", msg, err)
	}
}

func main() {
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	defer conn.Close()

	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	q, err := ch.QueueDeclare(
		"hello", // name
		false,   // durable
		false,   // delete when unused
		false,   // exclusive
		false,   // noWait
		nil,     // arguments
	)
	failOnError(err, "Failed to declare a queue")

	// Define a new context with a given timeout, that will be use by PublishWithContext
	// For context package, see https://pkg.go.dev/context and https://go.dev/blog/context
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	// Cancel context
	defer cancel()

	body := bodyFrom(os.Args)
	err = ch.PublishWithContext(ctx,
		"",     // exchange
		q.Name, // routing key
		false,  // mandatory
		false,
		amqp.Publishing{
			ContentType: "text/plain",
			Body:        []byte(body),
		})
	failOnError(err, "Failed to publish a message")
	log.Printf(" [x] Sent %s", body)

}

func bodyFrom(args []string) string {
	var s string
	if (len(args) < 2) || os.Args[1] == "" {
		s = "hello"
	} else {
		s = strings.Join(args[1:], " ")
	}
	return s
}
