// Consumer that receives tasks from a queue managed by RabbitMQ and performs them.
// Goal: distribute time-consuming tasks among multiple workers.
// See https://www.rabbitmq.com/tutorials/tutorial-two-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://pkg.go.dev/github.com/rabbitmq/amqp091-go
// Incremental development - 1st version (v1): receive the task and perform it.
// The number of dots in the message represents the task complexity;
// we simulate the task execution by sleeping for as many seconds as the number
// of dots in the message body.
// Use this consumer with new_task_v1.go

package main

import (
	"bytes"
	"log"
	"time"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Fatalf("%s: %s", msg, err)
	}
}

func main() {
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	defer conn.Close()

	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	q, err := ch.QueueDeclare(
		"hello", // name
		false,   // durable
		false,   // delete when unused
		false,   // exclusive
		false,   // noWait
		nil,     // arguments
	)
	failOnError(err, "Failed to declare a queue")

	msgs, err := ch.Consume(
		q.Name, // queue
		"",     // consumer
		true,   // auto-ack
		false,  // exclusive
		false,  // no-local
		false,  // no-wait
		nil,    // optional arguments
	)
	failOnError(err, "Failed to register a consumer")

	// Use channel forever to let main() wait, so it does not terminate
	forever := make(chan bool)

	go func() {
		for d := range msgs {
			log.Printf("Received a message: %s", d.Body)
			dot_count := bytes.Count(d.Body, []byte("."))
			t := time.Duration(dot_count)
			time.Sleep(t * time.Second)
			log.Printf("Done")
		}
	}()

	log.Printf(" [*] Waiting for messages. To exit press CTRL+C")
	<-forever
}
