// Consumer that receives tasks from a queue managed by RabbitMQ and performs them.
// Goal: distribute time-consuming tasks among multiple workers.
// See https://www.rabbitmq.com/tutorials/tutorial-two-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://pkg.go.dev/github.com/rabbitmq/amqp091-go
// Incremental development - 2nd version (v2): use message ack to tell RabbitMQ server
// that a particular message has been received, processed and that RabbitMQ is free to delete it.
// Use of acks guarantees at-least-once delivery. Without acks, message loss can happen
// during publish and consume operations and at-most-once delivery is guaranteed.
// Use this consumer with new_task_v1.go

package main

import (
	"bytes"
	"log"
	"time"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Fatalf("%s: %s", msg, err)
	}
}

func main() {
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	defer conn.Close()

	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	q, err := ch.QueueDeclare(
		"hello", // name
		false,   // durable
		false,   // delete when unused
		false,   // exclusive
		false,   // no-wait
		nil,     // arguments
	)
	failOnError(err, "Failed to declare a queue")

	msgs, err := ch.Consume(
		q.Name, // queue
		"",     // consumer
		false,  // autoAck: changed from v1
		false,  // exclusive
		false,  // noLocal
		false,  // noWait
		nil,    // optional arguments
	)
	failOnError(err, "Failed to register a consumer")

	// Use channel forever to let main() wait, so it does not terminate
	forever := make(chan bool)

	go func() {
		for d := range msgs {
			log.Printf("Received a message: %s", d.Body)
			dot_count := bytes.Count(d.Body, []byte("."))
			t := time.Duration(dot_count)
			time.Sleep(t * time.Second)
			log.Printf("Done")
			d.Ack(false) // added: acknowledge a single delivery once the task is processed
		}
	}()

	log.Printf(" [*] Waiting for messages. To exit press CTRL+C")
	<-forever
}
