// Consumer that receives tasks from a queue managed by RabbitMQ and performs them.
// Goal: distribute time-consuming tasks among multiple workers.
// See https://www.rabbitmq.com/tutorials/tutorial-two-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://pkg.go.dev/github.com/rabbitmq/amqp091-go
// Incremental development - 3rd version (v3): exploit message durability so that
// tasks will not be lost if RabbitMQ server either crashes or stops.
// We need to define a new queue to enable durability.
// Use this consumer with new_task_v3.go

package main

import (
	"bytes"
	"log"
	"time"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Fatalf("%s: %s", msg, err)
	}
}

func main() {
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	defer conn.Close()

	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	// Since RabbitMQ doesn't allow us to redefine an existing queue with different
	// parameters, we need to declare a queue with different name.
	q, err := ch.QueueDeclare(
		"task_queue", // name: changed
		true,         // durable: changed (now set to true)
		false,        // delete when unused
		false,        // exclusive
		false,        // noWait
		nil,          // optional arguments
	)
	failOnError(err, "Failed to declare a queue")

	msgs, err := ch.Consume(
		q.Name, // queue
		"",     // consumer
		false,  // autoAck
		false,  // exclusive
		false,  // noLocal
		false,  // noWait
		nil,    // arguments
	)
	failOnError(err, "Failed to register a consumer")

	// Use channel forever to let main() wait, so it does not terminate
	forever := make(chan bool)

	go func() {
		for d := range msgs {
			log.Printf("Received a message: %s", d.Body)
			dot_count := bytes.Count(d.Body, []byte("."))
			t := time.Duration(dot_count)
			time.Sleep(t * time.Second)
			log.Printf("Done")
			d.Ack(false)
		}
	}()

	log.Printf(" [*] Waiting for messages. To exit press CTRL+C")
	<-forever
}
