// Consumer that receives tasks from a queue managed by RabbitMQ and performs them.
// Goal: distribute time-consuming tasks among multiple workers.
// See https://www.rabbitmq.com/tutorials/tutorial-two-go.html
// Use Go RabbitMQ Client Library: an AMQP 0.9.1 client with RabbitMQ extensions in Go
// Go RabbitMQ Client Library: see https://pkg.go.dev/github.com/rabbitmq/amqp091-go
// Incremental development - 4th version (v4): exploit fair dispatching among multiple
// consumers by looking at the number of unacknowledged messages for each consumer,
// so to not dispatch a new message to a worker until it has processed and acknowledged the previous one.
// Use this consumer with new_task_v3.go

package main

import (
	"bytes"
	"log"
	"time"

	amqp "github.com/rabbitmq/amqp091-go"
)

func failOnError(err error, msg string) {
	if err != nil {
		log.Fatalf("%s: %s", msg, err)
	}
}

func main() {
	conn, err := amqp.Dial("amqp://guest:guest@localhost:5672/")
	failOnError(err, "Failed to connect to RabbitMQ")
	defer conn.Close()

	ch, err := conn.Channel()
	failOnError(err, "Failed to open a channel")
	defer ch.Close()

	q, err := ch.QueueDeclare(
		"task_queue", // name
		true,         // durable: changed
		false,        // delete when unused
		false,        // exclusive
		false,        // noWait
		nil,          // arguments
	)
	failOnError(err, "Failed to declare a queue")

	err = ch.Qos( // added: we set the prefetch count to 1
		1,     // prefetch count
		0,     // prefetch size
		false, // global
	)
	failOnError(err, "Failed to set QoS")

	msgs, err := ch.Consume(
		q.Name, // queue
		"",     // consumer
		false,  // auto-ack
		false,  // exclusive
		false,  // no-local
		false,  // no-wait
		nil,    // optional arguments
	)
	failOnError(err, "Failed to register a consumer")

	// Use channel forever to let main() wait, so it does not terminate
	forever := make(chan bool)

	// Consume messages in a goroutine
	go func() {
		for d := range msgs {
			log.Printf("Received a message: %s", d.Body)
			dot_count := bytes.Count(d.Body, []byte("."))
			t := time.Duration(dot_count)
			time.Sleep(t * time.Second)
			log.Printf("Done")
			d.Ack(false)
		}
	}()

	log.Printf(" [*] Waiting for messages. To exit press CTRL+C")
	<-forever
}
